import { defineComponent, PropType, Ref, ref, watch } from 'vue';
import { i18n } from 'src/boot/i18n';
import { isEmail } from '../../../boot/functions';
import { isMobilePhone } from '../../../boot/functions';
import { CustomerLevelType, CustomerType } from 'src/assets/type';

export default defineComponent({
  props: {
    showDialog: {
      type: Boolean,
      required: true,
    },
    levelOptions: {
      type: Array as PropType<CustomerLevelType[]>,
      required: true,
    },
    listCustomers: {
      type: Array as PropType<CustomerType[]>,
      required: true,
    },
  },
  setup(props, context) {
    watch(
      () => props.showDialog,
      (value) => {
        if (value) {
          void resetData();
        }
      }
    );
    const code: Ref<string | null> = ref(null);
    const userName: Ref<string | null> = ref(null);
    const fullName: Ref<string | null> = ref(null);
    const companyName: Ref<string | null> = ref(null);
    const taxCode: Ref<string | null> = ref(null);
    const email: Ref<string | null> = ref(null);
    const status: Ref<number | null> = ref(1);
    const phone: Ref<string | null> = ref(null);
    const password: Ref<string | null> = ref(null);
    const address: Ref<string | null> = ref(null);
    const type: Ref<string | null> = ref(null);
    const representative: Ref<string | null> = ref(null);
    const position: Ref<string | null> = ref(null);
    const level: Ref<CustomerLevelType | null> = ref(null);

    const confirmAddCustomer = () => {
      context.emit('addNewCustomer', {
        code: code.value,
        userName: userName.value,
        fullName: fullName.value,
        companyName: companyName.value,
        taxCode: taxCode.value,
        email: email.value,
        status: status.value,
        phone: phone.value,
        password: password.value,
        address: address.value,
        type: type.value,
        representative: representative.value,
        position: position.value,
        level: level.value,
      });
    };
    const resetData = () => {
      code.value = null;
      userName.value = null;
      fullName.value = null;
      companyName.value = null;
      taxCode.value = null;
      email.value = null;
      status.value = 1;
      phone.value = null;
      password.value = null;
      address.value = null;
      type.value = null;
      representative.value = null;
      position.value = null;
      level.value = null;
    };
    const userNameRules = [
      (val?: string) =>
        (val && val.trim().length) ||
        i18n.global.t('customer.validateMessages.requireUserName'),
      (val?: string) =>
        (val && val.trim().length < 30) ||
        i18n.global.t('customer.validateMessages.lengthUserName'),
    ];
    const passwordRules = [
      (val?: string) =>
        (val && val.trim().length) ||
        i18n.global.t('customer.validateMessages.requirePassword'),
      (val?: string) =>
        (val && val.trim().length >= 8) ||
        i18n.global.t('customer.validateMessages.lengthPassword'),
    ];
    const codeRules = [
      (val?: string) =>
        (val && val.trim().length) ||
        i18n.global.t('customer.validateMessages.requireCode'),
      (val?: string) =>
        (val && val.trim().length <= 50) ||
        i18n.global.t('customer.validateMessages.lengthCode'),
    ];
    const customerNameRules = [
      (val?: string) =>
        (val && val.trim().length) ||
        i18n.global.t('customer.validateMessages.requireCustomerName'),
      (val?: string) =>
        (val && val.trim().length <= 50) ||
        i18n.global.t('customer.validateMessages.lengthCustomerName'),
    ];
    const businessNameRules = [
      (val?: string) =>
        (val && val.trim().length) ||
        i18n.global.t('customer.validateMessages.requireBusinessName'),
    ];
    const taxCodeRules = [
      (val?: string) =>
        (val && val.trim().length) ||
        i18n.global.t('customer.validateMessages.requireTaxCode'),
      (val?: string) =>
        (val && val.trim().length <= 20) ||
        i18n.global.t('customer.validateMessages.LengthTaxCode'),
    ];
    const phoneRules = [
      (val?: string) =>
        (val && val.trim().length) ||
        i18n.global.t('customer.validateMessages.requirePhone'),
      (val: string) =>
        isMobilePhone(val) ||
        i18n.global.t('customer.validateMessages.isPhone'),
    ];
    const emailRules = [
      (val?: string) =>
        (val && val.trim().length) ||
        i18n.global.t('customer.validateMessages.requireEmail'),
      (val: string) =>
        isEmail(val) || i18n.global.t('customer.validateMessages.isEmail'),
    ];
    const addressRules = [
      (val?: string) =>
        (val && val.trim().length) ||
        i18n.global.t('customer.validateMessages.requireAddress'),
    ];
    const businessTypeRules = [
      (val?: string) =>
        (val && val.trim().length) ||
        i18n.global.t('customer.validateMessages.requireBusinessType'),
    ];
    const levelRules = [
      (val?: CustomerLevelType) =>
        val !== null ||
        i18n.global.t('customer.validateMessages.requireRatings'),
    ];
    const representativeRules = [
      (val?: string) =>
        (val && val.trim().length) ||
        i18n.global.t('customer.validateMessages.requireRepresentative'),
      (val?: string) =>
        (val && val.trim().length <= 50) ||
        i18n.global.t('customer.validateMessages.lengthRepresentative'),
    ];
    const positionRules = [
      (val?: string) =>
        (val && val.trim().length) ||
        i18n.global.t('customer.validateMessages.requiredPosition'),
    ];
    return {
      isPwd: ref(true),
      userNameRules,
      customerNameRules,
      businessNameRules,
      passwordRules,
      codeRules,
      taxCodeRules,
      emailRules,
      levelRules,
      addressRules,
      businessTypeRules,
      representativeRules,
      positionRules,
      phoneRules,
      userName,
      fullName,
      companyName,
      taxCode,
      email,
      status,
      phone,
      password,
      address,
      type,
      representative,
      position,
      level,
      code,
      confirmAddCustomer,
    };
  },
  emits: ['update:showDialog', 'click:CloseBtn', 'addNewCustomer'],
});
